/************************************************************************
*                       STREAMS Configuration                           *
*************************************************************************
*									*
* This program processes an ASCII configuration file and builds the	*
* modconf.c file containing basic configuration entries for each	*
* STREAMS driver or module.						*
*									*
* The syntax of the config file is as follows:				*
*									*
* Lines beginning with '#' are comments.				*
* Blank lines are ignored.						*
*									*
* Each significant line is introduced by a keyword which describes	*
* the type of entry it is and determines the syntax of the rest		*
* of the line.								*
*									*
*									*
* Driver specification:							*
*									*
*	driver	<driver-name> <prefix> <major>	<minors>		*
*									*
* The keyword "driver" introduces this entry.				*
*									*
* <driver-name> is the name of the driver, for example loop-around.	*
* It is any string of characters not including "white space" chars.	*
*									*
* <prefix> is alphanumeric, plus underscore, and will be prefixed	*
* to any table entries referencing variables within this driver.	*
* For example, the prefix loop_ will be used to derive the reference	*
* to the streamtab entry loop_info.  The driver must have a variable	*
* declared within it by the name <prefix>info of type struct streamtab.	*
*									*
* <major> specifies the major device number that will be assigned to	*
* the device.  The character "*" means that the config program will	*
* choose the major number.  A numeric value is used as it is given.	*
*									*
* <minors> is the number of minors the device has.  If this field is	*
* omitted, the number 1 will be used.					*
*									*
* Example:								*
*									*
*	driver loop-around loop_ *					*
*									*
*									*
* Module specification:							*
*									*
*	module	<name> <prefix>						*
*									*
* The keyword "module" introduces this entry.				*
*									*
* <name> is the name of the module, for example relay-mod.		*
* It is any string of characters not including "white space" chars.	*
*									*
* <prefix> is alphanumeric, plus underscore, and will be prefixed	*
* to any table entries referencing variables within this module.	*
* For example, the prefix relay_ will be used to derive the reference	*
* to the streamtab entry relay_info.  The module must have a variable	*
* declared within it by the name <prefix>info of type struct streamtab.	*
*									*
* Example:								*
*									*
*	module relay-mod relay_						*
*									*
*									*
* Node specification:							*
*									*
*	node	<dev-name> <type> <perm> <major> <minor>		*
*									*
* The keyword "node" introduces this entry.				*
*									*
* <dev-name> is the full path name of the device file to be created,	*
* for example, /dev/loop.						*
*									*
* <type> is the type of file to create. This consists of a single	*
* letter as follows:							*
*									*
*	b	block special device (S_IFBLK)				*
*	c	character special device (S_IFCHR)			*
*	p	pipe device (S_IFIFO)					*
*									*
* <perm> is the device permissions to use, 0666 gives everyone		*
* permission to use the device.  <perm> can be "*" in which case	*
* default permissions are supplied.					*
*									*
* <major> is the major device number to assign to this entry.  It can	*
* be a number, but is usually the <name> of the associated driver	*
* entry.  The major specified for the driver is then used.		*
*									*
* <minor> is the minor device number to assign to this entry.  It can	*
* be a number or the name of another driver.  If the name form is used	*
* then the major number of the other driver is used as the minor number	*
* of the device being created.						*
*									*
* Example:								*
*									*
*	node /dev/loop 		c * loop-around 1			*
*	node /dev/loop_clone	c * clone	loop-around		*
*									*
*									*
* Device specifications:						*
*									*
* These entries are used to describe the configuration of each		*
* physical device or port of a multi-port device on the system.		*
* The syntax for this type of entry is as follow:			*
*									*
*	device <driver-name> <unit> <port> <nports> <irq-share> <irq>	*
*			     <mem> <mem-size> <dma1> <dma2>		*
*									*
* All 11 fields must be on the same line.				*
*									*
* <driver-name> is the name of the driver that handles the device.	*
* It must match the name of a declared driver in this file.		*
*									*
* <unit> is the unit number of the device being declared.  A unit	*
* number means anything that the driver writer wants it to.  It is	*
* assumed to be a small integer that will index a configuration table	*
* that is built from #defines generated by running this program.	*
*									*
* <port> is the I/O port that the device uses.  Set to -1 if the	*
* device does not use any I/O ports.					*
*									*
* <nports> is the number of consecutive I/O ports used by the driver.	*
* If <ports> is -1 then <nports> can be any value, zero is suggested.	*
*									*
* <irq-share> is a letter-coded field as follows:			*
*									*
*	n	No sharing of IRQ					*
*	s	Sharing of IRQ						*
*	S	Sharing of IRQ but only with the same driver		*
*	x	Don't care, the device does not use an IRQ		*
*									*
* <irq> is the interrupt number for the device.  The value -1 is used	*
* when the device does not have an interrupt.  If this value is		*
* specified then it is assumed that there exists a routine with 	*
* the name <prefix>intr, where <prefix> is the driver prefix specified	*
* in the driver section.						*
*									*
* <mem> is the shared-memory address used by this device, if any.	*
* Set to 0 if the device does not use shared memory.			*
*									*
* <mem-size> is the amount of shared memory used, in bytes.		*
*									*
* <dma1> and <dma2> specify up to two DMA channels used by this		*
* device.  Set to -1 if no DMA channels are being used.			*
*									*
* The device entry causes a set of #defines to be generated into	*
* the file config.h.  This file may be #included into a driver		*
* configuration file to load tables, etc.				*
*									*
*									*
* Driver Initialization Specification:					*
*									*
* This entry specifies that a driver has an initialization function	*
* that needs to be called when STREAMS is initialized.  This occurs	*
* at system boot time just after the driver is registered.		*
*									*
* The syntax is:							*
*									*
*	initialize <driver-name>					*
*									*
* The keyword "initialize" introduces this specification.		*
*									*
* The <driver-name> is the name of the driver whose init routine is	*
* to be called.  The driver must supply a routine such as the following:*
*									*
* 	void <prefix>init(void)						*
*	{								*
*	}								*
*									*
* Example:								*
*									*
*	initialize loop-around						*
*									*
* This leads to a call to the routine loop_init() since "loop_" is the	*
* associated prefix for the loop-around driver.				*
*									*
*									*
* Tunable Parameter Specification:					*
*									*
* This type of entry allows you to insert define text into the		*
* output file (config.h default).  These values can be used as		*
* parameters for drivers or the streams package itself.			*
*									*
* The syntax is:							*
*									*
*	define <name> <value>						*
*									*
* The <name> is the identifier that is being defined.  It must be	*
* constructed of characters that comprise a legal C language		*
* identifier.								*
*									*
* The <value> is any text up to the end of the line.			*
*									*
* The output for this entry is of the form:				*
*									*
*	#define <name> <value>						*
*									*
* Example:								*
*									*
*	define	MAX_PORTS	16					*
*									*
* Generates the following in the output file:				*
*									*
*	#define	MAX_PORTS	16					*
*									*
* Command line syntax:							*
*									*
*	strconf <options> <input-file>					*
*									*
* <options>								*
*									*
* -b<number>	Set the base major number for automatic assignment	*
*		of major device numbers.				*
*									*
* -h<file>	Set the output header file name to something other than	*
*		the default "config.h".					*
*									*
* -o<file>	Set the output file name rather than the default	*
*		name "modconf.c".  This is the file that must be	*
*		linked in with STREAMS.					*
*									*
* -m<file>	Set the output file name for the program that will	*
*		make the nodes to something other than "makenodes.c".	*
*		This program is compiled and then run at boot time	*
* 		to create all the /dev entries for the STREAMS drivers.	*
*									*
* -M<name>	The name of the mknod() routine to call.  The default	*
*		is "mknod".						*
*									*
* -p<perm>	Set default permissions to something other than 0666.	*
*									*
* -r<name>	The name of the routine to build in makenodes.c.	*
*		The default is "main".					*
*									*
*									*
* Author:	David Grothe <dave@gcom.com>				*
*									*
************************************************************************/
/*
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * 
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Cambridge,
 * MA 02139, USA.
 * 
 */

#ident "@(#) LiS strconf.c 1.6 11/28/97"

#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>
#include <stdarg.h>
#include <malloc.h>
#include <stdlib.h>
#ifdef QNX
#include <mem.h>
#else
#include <memory.h>
#include <string.h>
#endif




/************************************************************************
*                          Data Structures                              *
************************************************************************/

typedef struct di
{
    struct di	*link ;
    char	 name[50] ;
    char	 prefix[50] ;
    int		 major;
    int		 nminors ;
    int		 init ;			/* "true" if has init routine */

} driver_info_t ;

typedef struct mi
{
    struct mi	*link ;
    char	 name[50] ;
    char	 prefix[50] ;

} module_info_t ;

typedef struct ni
{
    struct ni	*link ;
    char	 name[100] ;
    int		 type ;			/* I_IFCHR, etc */
    int		 major;
    int		 minor ;

} node_info_t ;

typedef struct vi
{
    struct vi		*link ;
    char		 name[50] ;
    driver_info_t	*drvr ;
    int			 unit ;
    long		 port ;
    int			 nports ;
    int			 irq_share ;
    int			 irq ;
    long		 mem ;
    long		 mem_size ;
    int			 dma1 ;
    int			 dma2 ;

} device_info_t ;

#define	IRQ_SHARE_DC	0		/* x */
#define	IRQ_NO_SHARE	1		/* n */
#define	IRQ_SHARE	2		/* s */
#define	IRQ_SHARE_DEV	3		/* S */

typedef struct lk
{
    struct lk	*link ;
    char	 name[100] ;

} lookup_struct_t ;


/************************************************************************
*                         Storage Declarations                          *
************************************************************************/

int		next_major = 50 ;	/* larger than dedicated majors */

driver_info_t	*driver_head ;
module_info_t	*module_head ;
node_info_t	*node_head ;
device_info_t	*device_head ;
int		 ndevices ;

char		*outfile_name = "modconf.c" ;
char		*mkfile_name  = "makenodes.c" ;
char		*mknod_name   = "mknod" ;
char		*func_name    = "main" ;
char		*infile_name ;
char		*config_name  = "config.h" ;

FILE		*infile ;
FILE		*outfile ;
FILE		*mkfile ;
FILE		*configfile ;

char		 inbuf[600] ;
int		 line_nr ;
char		*nextp ;			/* nxt chr in buffer */
char		 token[200] ;			/* current input token */
long		 value ;			/* converted integer */
int		 class ;			/* token classification */
#define	NUMBER	1				/* numeric (converted) */
#define	IDENT	2				/* identifier */
#define	EOL	3				/* end of line */

int		 dflt_perm = 0666 ;		/* default permissions */

/************************************************************************
*                            link_on_end                                *
*************************************************************************
*									*
* link item onto the end of the list whose head is pointed to by 'head'	*
*									*
************************************************************************/
void	link_on_end(lookup_struct_t **head, lookup_struct_t *item)
{
    lookup_struct_t	*p ;

    item->link = NULL ;
    if (*head == NULL)
    {
	*head = item ;
	return ;
    }

    for (p = *head; p->link != NULL; p = p->link) ;

    p->link = item ;

} /* link_on_end */

/************************************************************************
*                           mk_driver                                   *
*************************************************************************
*									*
* Make an entry for a driver.						*
*									*
************************************************************************/
driver_info_t	*mk_driver(char *name, char *prefix, int maj, int nmin)
{
    driver_info_t	*p ;

    p = (driver_info_t *) malloc(sizeof(*p)) ;
    if (p == NULL)
    {
	fprintf(stderr, "Cannot allocate entry for \"%s\"\n", name) ;
	exit(1) ;
    }

    strncpy(p->name, name, sizeof(p->name)) ;
    strncpy(p->prefix, prefix, sizeof(p->prefix)) ;
    p->major = maj ;
    p->nminors = nmin ;
    p->init = 0 ;			/* default is no init routine */

    link_on_end((lookup_struct_t **) &driver_head, (lookup_struct_t *) p) ;

    return(p) ;

} /* mk_driver */

/************************************************************************
*                           mk_module                                   *
*************************************************************************
*									*
* Make an entry for a module.						*
*									*
************************************************************************/
module_info_t	*mk_module(char *name, char *prefix)
{
    module_info_t	*p ;

    p = (module_info_t *) malloc(sizeof(*p)) ;
    if (p == NULL)
    {
	fprintf(stderr, "Cannot allocate entry for \"%s\"\n", name) ;
	exit(1) ;
    }

    strncpy(p->name, name, sizeof(p->name)) ;
    strncpy(p->prefix, prefix, sizeof(p->prefix)) ;

    link_on_end((lookup_struct_t **) &module_head, (lookup_struct_t *) p) ;

    return(p) ;

} /* mk_module */

/************************************************************************
*                             mk_node                                   *
*************************************************************************
*									*
* Make an entry for a node.						*
*									*
************************************************************************/
node_info_t	*mk_node(char *name, int type, int maj, int min)
{
    node_info_t	*p ;

    p = (node_info_t *) malloc(sizeof(*p)) ;
    if (p == NULL)
    {
	fprintf(stderr, "Cannot allocate entry for \"%s\"\n", name) ;
	exit(1) ;
    }

    strncpy(p->name, name, sizeof(p->name)) ;
    p->type  = type ;
    p->major = maj ;
    p->minor = min ;

    link_on_end((lookup_struct_t **) &node_head, (lookup_struct_t *) p) ;

    return(p) ;

} /* mk_node */

/************************************************************************
*                             mk_device                                 *
*************************************************************************
*									*
* Make an entry for a device.						*
*									*
************************************************************************/
device_info_t	*mk_device(device_info_t *devp)
{
    device_info_t	*p ;

    p = (device_info_t *) malloc(sizeof(*p)) ;
    if (p == NULL)
    {
	fprintf(stderr, "Cannot allocate device entry for \"%s\"\n", devp->name) ;
	exit(1) ;
    }

    *p = *devp ;

    link_on_end((lookup_struct_t **) &device_head, (lookup_struct_t *) p) ;
    ndevices++ ;

    return(p) ;

} /* mk_device */

/************************************************************************
*                            lookup_name                                *
*************************************************************************
*									*
* This routine takes advantage of the fact that all of the table	*
* entries have the same first two fields:  a link and a name.		*
*									*
* It searches any of the tables for the named item.			*
*									*
************************************************************************/
void	*lookup_name(void *list_head, char *name)
{
    lookup_struct_t	*p ;

    for (p = (lookup_struct_t *) list_head; p != NULL; p = p->link)
	if (strcmp(name, p->name) == 0) return((void *) p) ;

    return(NULL) ;

} /* lookup_name */

/************************************************************************
*                           find_driver                                 *
*************************************************************************
*									*
* Find the driver by name.						*
*									*
************************************************************************/
driver_info_t	*find_driver(char *name)
{
    return((driver_info_t *) lookup_name(driver_head, name)) ;

} /* find_driver */

/************************************************************************
*                           find_module                                 *
*************************************************************************
*									*
* Find the module by name.						*
*									*
************************************************************************/
module_info_t	*find_module(char *name)
{
    return((module_info_t *) lookup_name(module_head, name)) ;

} /* find_module */

/************************************************************************
*                           find_node                                 *
*************************************************************************
*									*
* Find the node by name.						*
*									*
************************************************************************/
node_info_t	*find_node(char *name)
{
    return((node_info_t *) lookup_name(node_head, name)) ;

} /* find_node */

/************************************************************************
*                             err                                       *
*************************************************************************
*									*
* Print an error message.						*
*									*
************************************************************************/
void	err(char *fmt, ...)
{
    va_list args;

    fprintf(stderr, "\"%s\" line %d: ", infile_name, line_nr) ;
    va_start (args, fmt);
    vfprintf (stderr, fmt, args);
    va_end (args);

} /* err */

/************************************************************************
*                           read_line                                   *
*************************************************************************
*									*
* Read a line from infile.  Skip comment lines and blank lines.		*
* Return negative for EOF, return zero for success.			*
*									*
************************************************************************/
int	read_line(void)
{
    do
    {
	line_nr++ ;
	if (fgets(inbuf, sizeof(inbuf), infile) == NULL)
	{
	    strcpy(inbuf, "End-of-file") ;
	    return(-1) ;
	}

    } while (inbuf[0] == '#' || inbuf[0] == '\n') ;

    nextp = inbuf ;			/* set global ptr */
    return(0) ;

} /* read_line */

/************************************************************************
*                           get_next                                    *
*************************************************************************
*									*
* Get the next blank-delimited token and leave in the global token	*
* buffer.  Advance global input pointers.				*
*									*
* Return the class of the next thing, EOL if end of line.		*
*									*
************************************************************************/
int	get_next(void)
{
    char	*p ;
    int		 n ;

    p = token ;
    n = 0 ;

    while ( *nextp && *nextp <= ' ' ) nextp++ ;	/* skip white space */

    while ( *nextp && *nextp > ' ' && n < sizeof(token)-1 )
    {						/* copy token */
	*p++ = *nextp++ ;
	n++ ;
    }

    while ( *nextp && *nextp > ' ' ) nextp++ ;	/* skip excess chars */

    *p = 0 ;					/* NUL terminate the token */

    if (isdigit(token[0]))			/* numeric value */
    {
	class = NUMBER ;
	value = (int) strtol(token, NULL, 0) ;	/* input convert */
    }
    else
    if (token[0] == 0)				/* end of line */
    {
	strcpy(token, "End-of-line") ;
	class = EOL ;
    }
    else					/* treat as identifier */
	class = IDENT ;

    return(class) ;				/* return its class */

} /* get_next */

/************************************************************************
*                           process_line                                *
*************************************************************************
*									*
* Process one line of input.						*
*									*
************************************************************************/
void	process_line(void)
{
    if (get_next() == EOL)
    {
	err("Internal error:  empty line\n") ;
	return ;
    }

    if (strcmp(token, "driver") == 0)
    {
	char	 name[50] ;
	char	 prefix[50] ;
	int	 major;
	int	 nminors ;

	if (get_next() != IDENT)
	{
	    err("Driver name must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(name, token, sizeof(name)-1) ;
	name[sizeof(name)-1] = 0 ;

	if (get_next() != IDENT)
	{
	    err("Driver prefix must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(prefix, token, sizeof(prefix)-1) ;
	name[sizeof(prefix)-1] = 0 ;

	switch (get_next())
	{
	case NUMBER:
	    major = (int) value ;
	    break ;
	case IDENT:
	    if (strcmp(token, "*") == 0)
	    {
		major = next_major++ ;
		break ;
	    }
	    /*
	     * Fall into error case 
	     */
	default:
	    err("Invalid driver major specifier: %s\n", token) ;
	    return ;
	}

	switch (get_next())
	{
	case NUMBER:
	    nminors = (int) value ;
	    break ;
	case EOL:
	    nminors = 1 ;
	    break ;
	default:
	    err("Invalid driver n-minors specifier: %s\n", token) ;
	    return ;
	}

	mk_driver(name, prefix, major, nminors) ;	/* make the entry */
    }
    else
    if (strcmp(token, "module") == 0)
    {
	char	 name[50] ;
	char	 prefix[50] ;

	if (get_next() != IDENT)
	{
	    err("Module name must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(name, token, sizeof(name)-1) ;
	name[sizeof(name)-1] = 0 ;

	if (get_next() != IDENT)
	{
	    err("Module prefix must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(prefix, token, sizeof(prefix)-1) ;
	name[sizeof(prefix)-1] = 0 ;

	mk_module(name, prefix) ;
    }
    else
    if (strcmp(token, "node") == 0)
    {
	char	 	 name[100] ;
	int	 	 type ;			/* I_IFCHR, etc */
	int	 	 major;
	int	 	 minor ;
	driver_info_t	*dp ;

	if (get_next() != IDENT)
	{
	    err("Node name must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(name, token, sizeof(name)-1) ;
	name[sizeof(name)-1] = 0 ;

	switch (get_next())
	{
	case IDENT:
	    if (strcmp(token, "c") == 0)
	    {
		type = S_IFCHR ;
		break ;
	    }
	    else
	    if (strcmp(token, "b") == 0)
	    {
		type = S_IFBLK ;
		break ;
	    }
	    else
	    if (strcmp(token, "p") == 0)
	    {
		type = S_IFIFO ;
		break ;
	    }
	    /*
	     * Fall into error case 
	     */
	case NUMBER:
	case EOL:
	default:
	    err("Invalid device type specifier: %s\n", token) ;
	    return ;
	}

	switch (get_next())
	{
	case NUMBER:
	    type |= value ;			/* add permissions */
	    break ;

	case IDENT:
	    if (strcmp(token, "*") == 0)
	    {
		type |= dflt_perm ;
		break ;
	    }
	    /*
	     * Fall through to error
	     */
	case EOL:
	default:
	    err("Invalid permissions specifier: %s\n", token) ;
	    return ;
	}

	switch (get_next())
	{
	case IDENT:
	    dp = find_driver(token) ;
	    if (dp == NULL)
	    {
		err("Invalid driver name: %s\n", token) ;
		return ;
	    }

	    major = dp->major ;
	    break ;

	case NUMBER:
	    major = value ;
	    break ;
	case EOL:
	default:
	    err("Invalid device major specifier: %s\n", token) ;
	    return ;
	}

	switch (get_next())
	{
	case IDENT:
	    dp = find_driver(token) ;
	    if (dp == NULL)
	    {
		err("Invalid driver name: %s\n", token) ;
		return ;
	    }

	    minor = dp->major ;
	    break ;

	case NUMBER:
	    minor = value ;
	    break ;
	case EOL:
	default:
	    err("Invalid device minor specifier: %s\n", token) ;
	    return ;
	}

	mk_node(name, type, major, minor) ;
    }
    else
    if (strcmp(token, "device") == 0)
    {
	device_info_t	 dev ;

	memset(&dev, 0, sizeof(dev)) ;
	if (get_next() != IDENT)
	{
	    err("Driver name must be identifier: %s\n", token) ;
	    return ;
	}

	strncpy(dev.name, token, sizeof(dev.name)-1) ;
	dev.name[sizeof(dev.name)-1] = 0 ;
	dev.drvr = find_driver(token) ;
	if (dev.drvr == NULL)
	{
	    err("Driver name \"%s\" not found\n") ;
	    return ;
	}

	if (get_next() == NUMBER)
	    dev.unit = value ;
	else
	if (class == EOL)
	    dev.unit = -1 ;
	else
	{
	    err("Unit number must be a number: %s\n") ;
	    return ;
	}

	if (get_next() == NUMBER)
	    dev.port = value ;
	else
	if (class == EOL)
	    dev.port = -1 ;
	else
	{
	    err("Port address must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.nports = (int) value ;
	else
	if (class == EOL)
	    dev.nports = 0 ;
	else
	{
	    err("Number of port addresses must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == IDENT)
	    switch (token[0])
	    {
	    case 'n':
		dev.irq_share =	IRQ_NO_SHARE ;
		break ;
	    case 's':
		dev.irq_share =	IRQ_SHARE ;
		break ;
	    case 'S':
		dev.irq_share =	IRQ_SHARE_DEV ;
		break ;
	    case 'x':
		dev.irq_share =	IRQ_SHARE_DC ;
		break ;
	    default:
		err("Invalid IRQ sharing code: \"%s\"\n", token) ;
		return ;
	    }
	else
	if (class == EOL)
	    dev.irq_share = 0 ;
	else
	{
	    err("IRQ sharing code must be identifier: %s\n", token) ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.irq = (int) value ;
	else
	if (class == EOL)
	    dev.irq = -1 ;
	else
	{
	    err("IRQ must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.mem = value ;
	else
	if (class == EOL)
	    dev.mem = 0 ;
	else
	{
	    err("Memory address must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.mem_size = value ;
	else
	if (class == EOL)
	    dev.mem_size = 0 ;
	else
	{
	    err("Memory size must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.dma1 = (int) value ;
	else
	if (class == EOL)
	    dev.dma1 = -1 ;
	else
	{
	    err("DMA channel must be a number: %s\n") ;
	    return ;
	}


	if (get_next() == NUMBER)
	    dev.dma2 = (int) value ;
	else
	if (class == EOL)
	    dev.dma2 = -1 ;
	else
	{
	    err("DMA channel must be a number: %s\n") ;
	    return ;
	}

	mk_device(&dev) ;
    }
    else
    if (strcmp(token, "initialize") == 0)
    {
	driver_info_t	 *dp ;

	if (get_next() != IDENT)
	{
	    err("Driver name must be identifier: %s\n", token) ;
	    return ;
	}

	dp = find_driver(token) ;
	if (dp == NULL)
	{
	    err("Driver name \"%s\" not found\n") ;
	    return ;
	}

	dp->init = 1 ;			/* driver has init flag */
    }
    else
    if (strcmp(token, "define") == 0)
    {
	fprintf(configfile, "#define %s\n", nextp) ;
    }
    else
	err("Invalid keyword \"%s\"\n", token) ;

} /* process_line */

/************************************************************************
*                           process_file                                *
*************************************************************************
*									*
* Process the input file.						*
*									*
************************************************************************/
void	process_file(void)
{
    while (read_line() == 0)
    {
	process_line() ;
    }

} /* process_file */

/************************************************************************
*                           build_modconf                               *
*************************************************************************
*									*
* Build the modconf.c file.						*
*									*
************************************************************************/
void	build_modconf(void)
{
    driver_info_t	*dp ;
    module_info_t	*mp ;

#define	p0(f)	fprintf(outfile, f)
#define	p1(f,y)	fprintf(outfile, f,y)

    p0("/* WARNING:  THIS FILE IS PROGRAMATICALLY GENERATED. */\n");
    p0("/*           DO NOT MODIFY BY HAND.                  */\n");
    p0("\n") ;
    p0("#include <sys/stream.h>\n") ;
    p0("\n") ;

    for (dp = driver_head; dp != NULL; dp = dp->link)
    {
	p1("extern struct streamtab		%sinfo ;\n", dp->prefix) ;
	if (dp->init)
	    p1("extern void				%sinit(void) ;\n", dp->prefix) ;

    }
    p0("\n") ;

    p0("driver_config_t		lis_driver_config[] =\n") ;
    p0("{\n") ;
    for (dp = driver_head; dp != NULL; dp = dp->link)
    {
	p1("	{\"%s\",", dp->name) ;
	p1(" &%sinfo,", dp->prefix) ;
	p1(" %d,", dp->major) ;
	p1(" %d,", dp->nminors) ;
	if (dp->init)
	    p1(" %sinit},\n", dp->prefix) ;
	else
	    p0(" NULL},\n") ;
    }
    p0("\n") ;
    p0("	{\"\", NULL, 0, 0, NULL} /* always the final element */\n") ;
    p0("} ;\n") ;
    p0("\n") ;

    for (mp = module_head; mp != NULL; mp = mp->link)
    {
	p1("extern struct streamtab		%sinfo ;\n", mp->prefix) ;
    }
    p0("\n") ;

    p0("module_config_t		lis_module_config[] =\n") ;
    p0("{\n") ;
    for (mp = module_head; mp != NULL; mp = mp->link)
    {
	p1("	{\"%s\",", mp->name) ;
	p1(" &%sinfo},\n", mp->prefix) ;
    }
    p0("\n") ;
    p0("	{\"\", NULL}		/* always the final element */\n") ;
    p0("} ;\n") ;

    fclose(outfile) ;

#undef p0
#undef p1

} /* build_modconf */

/************************************************************************
*                           build_mknods				*
*************************************************************************
*									*
* Build the makenodes.c file.						*
*									*
************************************************************************/
void	build_mknods(void)
{
    node_info_t	*dp ;

#define	p0(f)	fprintf(mkfile, f)
#define	p1(f,y)	fprintf(mkfile, f,y)

    p0("/* WARNING:  THIS FILE IS PROGRAMATICALLY GENERATED. */\n");
    p0("/*           DO NOT MODIFY BY HAND.                  */\n");
    p0("\n") ;
    p0("#include <sys/types.h>\n") ;
    p0("#include <sys/stat.h>\n") ;
    p0("#include <sys/stream.h>\n") ;
    p0("#include <stdio.h>\n") ;
    p0("#include <fcntl.h>\n") ;
    p0("#include <unistd.h>\n") ;
    p0("\n") ;

    p1("void %s(void)\n", func_name) ;
    p0("{\n") ;

    p0("	int	rslt ;\n") ;
    p0("\n") ;
    p0("	(void)umask(0)\n;");
    p0("\n") ;
    for (dp = node_head; dp != NULL; dp = dp->link)
    {
	p1("	(void)unlink(\"%s\");\n", dp->name) ;
	p1("	rslt = %s(", mknod_name) ;
	p1("\"%s\", ", dp->name) ;
	p1("0%o, ", dp->type) ;
	p1("makedevice(%d,", dp->major) ;
	p1("%d)) ;\n", dp->minor) ;
	p1("	if (rslt < 0) printf(\"%s: %%s\\n\", strerror(-rslt));\n",
			    dp->name) ;
	p0("\n") ;
    }

    p0("}\n") ;

    fclose(mkfile) ;

#undef p0
#undef p1

} /* build_mknods */

/************************************************************************
*                            raise                                      *
*************************************************************************
*									*
* Return the upper-case equivalent of the passed-in string.		*
*									*
************************************************************************/
char	*raise(char *p)
{
    static char		 uc[200] ;
    char		*ucp ;


    for (ucp = uc; *p; p++, ucp++) *ucp = toupper(*p) ;
    *ucp = 0 ;

    return(uc) ;

} /* raise */

/************************************************************************
*                             build_config                              *
*************************************************************************
*									*
* Build the file config.h.						*
*									*
************************************************************************/

#define	p0(f)		fprintf(configfile, f)
#define	p1(f,y)		fprintf(configfile, f,y)
#define	p2(f,y,z)	fprintf(configfile, f,y,z)

void	start_config(void)
{
    p0("/* WARNING:  THIS FILE IS PROGRAMATICALLY GENERATED. */\n");
    p0("/*           DO NOT MODIFY BY HAND.                  */\n");
    p0("\n") ;

} /* start_config */

void	build_config(void)
{
    driver_info_t	*drp ;
    device_info_t	*dp ;
    int			 devnr = 0 ;

    for (drp = driver_head; drp != NULL; drp = drp->link)
    {
	if (strcmp(drp->prefix, "-") == 0)	/* no name */
	    continue ;				/* no defines */

	p1("#define\t%s\t\t1\n", raise(drp->prefix));
	p2("#define\t%s_CNTLS\t%d\n", raise(drp->prefix), drp->nminors);
	p2("#define\t%s_UNITS\t%d\n", raise(drp->prefix), drp->nminors);
	p2("#define\t%s_CMAJORS\t%d\n", raise(drp->prefix), 1);
	p2("#define\t%s_CMAJOR_0\t%d\n", raise(drp->prefix), drp->major);
	/* TBD: calculate SVR4 compatible type */
	p2("#define\t%s_TYPE\t%d\n", raise(drp->prefix), -1);
	p0("\n") ;
    }

    for (dp = device_head; dp != NULL; dp = dp->link)
    {
	if (strcmp(dp->drvr->prefix, "-") == 0)	/* no name */
	    continue ;				/* no defines */

	devnr = dp->unit ;
	p2("#define\t%s_%u\t1\n", raise(dp->drvr->prefix), devnr);

	if (dp->port > 0)
	{
	    p2("#define\t%s_%u_SIOA\t", raise(dp->drvr->prefix), devnr);
	    p1("0x%lx\n", dp->port) ;
	}
	else
	    p2("#define\t%s_%u_SIOA\t0\n", raise(dp->drvr->prefix), devnr);

	if (dp->nports > 0)
	{
	    p2("#define\t%s_%u_EIOA\t", raise(dp->drvr->prefix), devnr);
	    p1("0x%lx\n", dp->port+dp->nports-1) ;
	}
	else
	{
	    p2("#define\t%s_%u_EIOA\t", raise(dp->drvr->prefix), devnr);
	    p1("0x%lx\n", dp->port) ;
	}

	p2("#define\t%s_%u_VECT\t", raise(dp->drvr->prefix), devnr);
	p1("%d\n", dp->irq) ;

	if (dp->mem > 0)
	{
	    p2("#define\t%s_%u_SCMA\t", raise(dp->drvr->prefix), devnr);
	    p1("0x%lx\n", dp->mem) ;
	    p2("#define\t%s_%u_ECMA\t", raise(dp->drvr->prefix), devnr);
	    p1("0x%lx\n", dp->mem + dp->mem_size - 1) ;
	}
	else
	{
	    p2("#define\t%s_%u_SCMA\t0\n", raise(dp->drvr->prefix), devnr);
	    p2("#define\t%s_%u_ECMA\t0\n", raise(dp->drvr->prefix), devnr);
	}

	p2("#define\t%s_%u_CHAN\t", raise(dp->drvr->prefix), devnr);
	p1("%d\n", dp->dma1) ;
	p2("#define\t%s_%u_CHAN2\t", raise(dp->drvr->prefix), devnr);
	p1("%d\n", dp->dma2) ;

	/* TBD:  come up with something for driver "type" */
	p0("\n") ;
    }

   fclose(configfile) ;

#undef p0
#undef p1
#undef p2

} /* build_config */

/************************************************************************
*                            print_options                              *
************************************************************************/
void	print_options(void)
{

#define	p	printf

p("Usage:  strconf <options> <input-file>\n");
p("\n");
p("<options>\n") ;
p("\n");
p("-b<number>   Set the base major number for automatic assignment\n");
p("             of major device numbers.\n");
p("-h<file>     Set the output header file name to something other than\n");
p("             the default \"config.h\".\n");
p("-o<file>     Set the output file name rather than the default\n");
p("             name \"modconf.c\".  This is the file that must be\n");
p("             linked in with STREAMS.\n");
p("-m<file>     Set the output file name for the program that will\n");
p("             make the nodes to something other than \"makenodes.c\".\n");
p("             This program is compiled and then run at boot time\n");
p("             to create all the /dev entries for the STREAMS drivers.\n");
p("-M<name>     The name of the mknod() routine to call.  The default\n");
p("             is \"mknod\".\n");
p("-p<perm>     Set default permissions to something other than 0666.\n");
p("-r<name>     The name of the routine to build in makenodes.c.\n");
p("             The default is \"main\".\n") ;

#undef p

} /* print_options */

/************************************************************************
*                          get_options                                  *
*************************************************************************
*									*
* Get command line options.						*
*									*
************************************************************************/
void	get_options(int argc, char **argv)
{
    char	*p ;

    for (argc--, argv++; argc > 0; argc--, argv++)
    {
	p = *argv ;
	if (*p == '-')
	    switch (*++p)
	    {
	    case 'b':			/* -b<number> */
		next_major = (int) strtol(p+1, NULL, 0) ;
		break ;
	    case 'h':			/* -h<file> */
		config_name = p+1 ;
		break ;
	    case 'o':			/* -o<file> */
		outfile_name = p+1 ;
		break ;
	    case 'm':			/* -m<file> */
		mkfile_name = p+1 ;
		break ;
	    case 'M':			/* -M<name> */
		mknod_name = p+1 ;
		break ;
	    case 'p':
		dflt_perm = (int) strtol(p+1, NULL, 0) ;
		break ;
	    case 'r':
	    	func_name = p+1 ;	/* -p<name> */
		break ;
	    default:
		print_options() ;
		exit(1) ;
	    }
	else
	    infile_name = p ;
    }

    if (infile_name == NULL || outfile_name == NULL || mkfile_name == NULL ||
        mknod_name == NULL || config_name == NULL)
    {
	print_options() ;
	exit(1) ;
    }

} /* get_options */

/************************************************************************
*                              main                                     *
************************************************************************/
void	main(int argc, char **argv)
{
    get_options(argc, argv) ;

    /*
     * Open all the files
     */
    infile = fopen(infile_name, "r") ;
    if (infile == NULL)
    {
	perror(infile_name) ;
	exit(1) ;
    }

    outfile = fopen(outfile_name, "w") ;
    if (outfile == NULL)
    {
	perror(outfile_name) ;
	exit(1) ;
    }

    mkfile = fopen(mkfile_name, "w") ;
    if (mkfile == NULL)
    {
	perror(mkfile_name) ;
	exit(1) ;
    }

    configfile = fopen(config_name, "w") ;
    if (configfile == NULL)
    {
	perror(config_name) ;
	exit(1) ;
    }

    start_config() ;
    process_file() ;
    build_modconf() ;
    build_mknods() ;
    build_config() ;

} /* main */
